"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OIDC_ROUTES = void 0;
exports.defineRoutes = defineRoutes;
exports.loginHandler = loginHandler;
var _configSchema = require("@kbn/config-schema");
var _sanitize_next_url = require("../../sanitize_next_url");
var _constants = require("../../../../../utils/constants");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const OIDC_ROUTES = exports.OIDC_ROUTES = {
  LOGIN: `${_constants.APP_ROOT}/auth/oidc/login`
};
function defineRoutes({
  authInstance,
  kibanaCore,
  kibanaConfig,
  logger,
  debugLog,
  searchGuardBackend
}) {
  const config = kibanaConfig;
  const basePath = kibanaCore.http.basePath.serverBasePath;
  const router = kibanaCore.http.createRouter();
  const routesPath = '/auth/oidc/';
  const loginSettings = {
    path: `${_constants.APP_ROOT}${routesPath}login`,
    validate: {
      query: _configSchema.schema.object({
        nextUrl: _configSchema.schema.maybe(_configSchema.schema.string()),
        // it comes from our login page
        next_url: _configSchema.schema.maybe(_configSchema.schema.string()) // it comes from the IDP login page
      }, {
        unknowns: 'allow'
      })
    },
    options: {
      authRequired: false
    }
  };
  const loginHandlerOptions = {
    basePath,
    kibanaCore,
    config,
    debugLog,
    authInstance,
    logger,
    searchGuardBackend
  };
  router.get(loginSettings, loginHandler(loginHandlerOptions));

  // Keep a POST route in case the IdP uses POSTs
  router.post(loginSettings, loginHandler(loginHandlerOptions));

  // We want to keep backwards compatibility with /auth/openid requests
  const legacyRoutesPath = '/auth/openid/';
  router.get({
    ...loginSettings,
    path: `${_constants.APP_ROOT}${legacyRoutesPath}login`
  }, loginHandler(loginHandlerOptions));
  router.post({
    ...loginSettings,
    path: `${_constants.APP_ROOT}${legacyRoutesPath}login`
  }, loginHandler(loginHandlerOptions));
} //end module

function loginHandler({
  basePath,
  config,
  authInstance,
  logger,
  searchGuardBackend,
  kibanaCore
}) {
  return async function (context, request, response) {
    const authCode = request.url.searchParams.get('code');
    if (!authCode) {
      return handleAuthRequest({
        request,
        response,
        basePath,
        config,
        searchGuardBackend,
        sessionStorageFactory: authInstance.sessionStorageFactory,
        logger
      });
    }

    // We have an auth code, now we need to try to exchange it for an id_token
    try {
      // Validate the state to make sure that the request was really
      // requested by Kibana in this session
      const sessionCookie = (await authInstance.sessionStorageFactory.asScoped(request).get()) || {};
      const cookieOidc = sessionCookie.oidc;
      if (!cookieOidc) {
        // This seems to happen when we have
        // a) No more session on the IdP
        // and b) We delete our cookie completely.
        // @todo Can we somehow restart the process here?
        throw new Error('OIDC request contained code, but we have no cookie content to compare with');
      }

      // Make sure to clear out what was used for this login request.
      delete sessionCookie.oidc;
      await authInstance.sessionStorageFactory.asScoped(request).set(sessionCookie);
      const credentials = {
        mode: 'oidc',
        sso_result: request.url.href,
        sso_context: cookieOidc.ssoContext,
        id: cookieOidc.authTypeId
      };

      // Authenticate with the token given to us by the IdP
      const authResponse = await authInstance.handleAuthenticate(request, credentials);
      let redirectTo = '/';
      if (authResponse.redirectUri) {
        redirectTo = (0, _sanitize_next_url.sanitizeNextUrl)(authResponse.redirectUri, basePath);
      }

      // All good, redirect to home
      return response.redirected({
        headers: {
          location: redirectTo
        }
      });
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta3, _error$meta3$body;
      logger.error('Error while trying to authenticate', error.meta ? JSON.stringify(error.meta) : error);
      var headers = {
        location: basePath + '/searchguard/login?err=oidc'
      };
      var cookies = [];
      if ((_error$meta = error.meta) !== null && _error$meta !== void 0 && (_error$meta$body = _error$meta.body) !== null && _error$meta$body !== void 0 && _error$meta$body.error) {
        var _error$meta2, _error$meta2$body;
        cookies.push('sg_err=' + encodeURIComponent((_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : _error$meta2$body.error) + "; Path=/");
      } else if (error.message) {
        cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
      }
      if ((_error$meta3 = error.meta) !== null && _error$meta3 !== void 0 && (_error$meta3$body = _error$meta3.body) !== null && _error$meta3$body !== void 0 && _error$meta3$body.debug) {
        var _error$meta4, _error$meta4$body;
        cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta4 = error.meta) === null || _error$meta4 === void 0 ? void 0 : (_error$meta4$body = _error$meta4.body) === null || _error$meta4$body === void 0 ? void 0 : _error$meta4$body.debug)) + "; Path=/");
      }
      if (cookies.length > 0) {
        headers['set-cookie'] = cookies;
      }
      return response.redirected({
        headers
      });
    }
  };
}

/**
 * Handle the first step of the process - obtain an auth code
 * @param request
 * @param response
 * @param sessionStorageFactory
 * @param clientId
 * @param redirectUri
 * @param nonce
 * @param scope
 * @returns {Promise<*>}
 */
async function handleAuthRequest({
  request,
  response,
  basePath,
  config,
  searchGuardBackend,
  sessionStorageFactory,
  logger
}) {
  // Add the nextUrl to the redirect_uri as a parameter. The IDP uses the redirect_uri to redirect the user after successful authentication.
  // For example, it is used to redirect user to the correct dashboard if the user put shared URL in the browser address input before authentication.
  // To make this work, append the wildcard (*) to the valid redirect URI in the IDP configuration, for example
  // https://kibana.example.com:5601/auth/oidc/login*
  let nextUrl = null;
  try {
    if (request.url.searchParams.get('nextUrl') && decodeURIComponent(request.url.searchParams.get('nextUrl')) !== '/') {
      // Do not add the nextUrl to the redirect_uri to avoid the following breaking
      // change for the users that use normal URL to authenticate.
      nextUrl = (0, _sanitize_next_url.sanitizeNextUrl)(request.url.searchParams.get('nextUrl'));
    }
  } catch (error) {
    // We may have received a malformed URL, caught by decodedURIComponent.
    // In this case we just proceed without a nextUrl.
  }
  let authConfig;
  const sessionCookie = (await sessionStorageFactory.asScoped(request).get()) || {};
  // We may have multiple OIDC configurations.
  // The authTypeId may still be in the cookie. This happens when
  // a session token expires and no explicit logout is made. We need
  // this behaviour so that we can "refresh" the credentials from the IdP.
  // (not to be confused with the OIDC refresh flow)

  const requestedAuthTypeId = request.url.searchParams.get('authTypeId') || sessionCookie.authTypeId;
  // Delete this again, otherwise the user won't get back to the login page
  // if trying to access Kibana again
  delete sessionCookie.authTypeId;
  delete sessionCookie.authType;
  const authConfigFinder = requestedAuthTypeId ? config => {
    return config.id === requestedAuthTypeId;
  } : config => {
    return config.method === 'oidc';
  };
  try {
    authConfig = (await searchGuardBackend.getAuthConfig(nextUrl)).auth_methods.find(authConfigFinder);
    if (!authConfig) {
      throw new Error('Auth config not found');
    }
  } catch (error) {
    var _error$meta5, _error$meta5$body, _error$meta7, _error$meta7$body;
    logger.error(`Error when trying to load the configuration for your IdP: ${error.stack}`);
    var headers = {
      location: basePath + '/login?err=oidc_init'
    };
    var cookies = [];
    if ((_error$meta5 = error.meta) !== null && _error$meta5 !== void 0 && (_error$meta5$body = _error$meta5.body) !== null && _error$meta5$body !== void 0 && _error$meta5$body.error) {
      var _error$meta6, _error$meta6$body;
      cookies.push('sg_err=' + encodeURIComponent((_error$meta6 = error.meta) === null || _error$meta6 === void 0 ? void 0 : (_error$meta6$body = _error$meta6.body) === null || _error$meta6$body === void 0 ? void 0 : _error$meta6$body.error) + "; Path=/");
    } else if (error.message) {
      cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
    }
    if ((_error$meta7 = error.meta) !== null && _error$meta7 !== void 0 && (_error$meta7$body = _error$meta7.body) !== null && _error$meta7$body !== void 0 && _error$meta7$body.debug) {
      var _error$meta8, _error$meta8$body;
      cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta8 = error.meta) === null || _error$meta8 === void 0 ? void 0 : (_error$meta8$body = _error$meta8.body) === null || _error$meta8$body === void 0 ? void 0 : _error$meta8$body.debug)) + "; Path=/");
    }
    if (cookies.length > 0) {
      headers['set-cookie'] = cookies;
    }
    return response.redirected({
      headers
    });
  }
  sessionCookie.oidc = {
    ssoContext: authConfig.sso_context,
    authTypeId: authConfig.id || null,
    query: {}
  };
  await sessionStorageFactory.asScoped(request).set(sessionCookie);
  return response.redirected({
    headers: {
      location: authConfig.sso_location
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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